<?php
require_once '../../config/database.php';
require_once '../includes/functions.php';

// Veritabanı bağlantısı
$pdo = getDBConnection();

// CRUD işlemleri
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add':
                $title = $_POST['title'];
                $content = $_POST['content'];
                $category = $_POST['category'];
                $author = $_POST['author'];
                $priority = $_POST['priority'];
                $tags = $_POST['tags'];
                $is_published = isset($_POST['is_published']) ? 1 : 0;
                $image_alt = $_POST['image_alt'] ?? '';
                
                $image_path = null;
                if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                    $uploadResult = uploadImage($_FILES['image']);
                    if ($uploadResult['success']) {
                        $image_path = $uploadResult['web_path'];
                    } else {
                        $error = $uploadResult['message'];
                        break;
                    }
                }
                
                $stmt = $pdo->prepare("INSERT INTO lesson_notes (title, content, category, author, priority, tags, is_published, image_path, image_alt) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$title, $content, $category, $author, $priority, $tags, $is_published, $image_path, $image_alt]);
                $success = "Ders notu başarıyla eklendi!";
                break;
                
            case 'edit':
                $id = $_POST['id'];
                $title = $_POST['title'];
                $content = $_POST['content'];
                $category = $_POST['category'];
                $author = $_POST['author'];
                $priority = $_POST['priority'];
                $tags = $_POST['tags'];
                $is_published = isset($_POST['is_published']) ? 1 : 0;
                $image_alt = $_POST['image_alt'] ?? '';
                
                // Mevcut resmi al
                $stmt = $pdo->prepare("SELECT image_path FROM lesson_notes WHERE id = ?");
                $stmt->execute([$id]);
                $old_image = $stmt->fetchColumn();
                
                $image_path = $old_image;
                
                // Yeni resim yüklendiyse
                if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                    $uploadResult = uploadImage($_FILES['image']);
                    if ($uploadResult['success']) {
                        // Eski resmi sil
                        if ($old_image) {
                            deleteImage($old_image);
                        }
                        $image_path = $uploadResult['web_path'];
                    } else {
                        $error = $uploadResult['message'];
                        break;
                    }
                }
                
                $stmt = $pdo->prepare("UPDATE lesson_notes SET title=?, content=?, category=?, author=?, priority=?, tags=?, is_published=?, image_path=?, image_alt=? WHERE id=?");
                $stmt->execute([$title, $content, $category, $author, $priority, $tags, $is_published, $image_path, $image_alt, $id]);
                $success = "Ders notu başarıyla güncellendi!";
                break;
                
            case 'delete':
                $id = $_POST['id'];
                
                // Önce resmi al ve sil
                $stmt = $pdo->prepare("SELECT image_path FROM lesson_notes WHERE id = ?");
                $stmt->execute([$id]);
                $image_path = $stmt->fetchColumn();
                
                if ($image_path) {
                    deleteImage($image_path);
                }
                
                $stmt = $pdo->prepare("DELETE FROM lesson_notes WHERE id=?");
                $stmt->execute([$id]);
                $success = "Ders notu başarıyla silindi!";
                break;
        }
    }
}

// Ders notlarını getir
$search = isset($_GET['search']) ? $_GET['search'] : '';
$category_filter = isset($_GET['category']) ? $_GET['category'] : '';
$priority_filter = isset($_GET['priority']) ? $_GET['priority'] : '';

$where_conditions = [];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(title LIKE ? OR content LIKE ? OR tags LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($category_filter)) {
    $where_conditions[] = "category = ?";
    $params[] = $category_filter;
}

if (!empty($priority_filter)) {
    $where_conditions[] = "priority = ?";
    $params[] = $priority_filter;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

$stmt = $pdo->prepare("SELECT * FROM lesson_notes $where_clause ORDER BY created_at DESC");
$stmt->execute($params);
$lesson_notes = $stmt->fetchAll();

// Kategorileri getir
$stmt = $pdo->query("SELECT DISTINCT category FROM lesson_notes ORDER BY category");
$categories = $stmt->fetchAll(PDO::FETCH_COLUMN);

// İstatistikler
$stats = [
    'total' => $pdo->query("SELECT COUNT(*) FROM lesson_notes")->fetchColumn(),
    'published' => $pdo->query("SELECT COUNT(*) FROM lesson_notes WHERE is_published = 1")->fetchColumn(),
    'categories' => $pdo->query("SELECT COUNT(DISTINCT category) FROM lesson_notes")->fetchColumn(),
    'this_month' => $pdo->query("SELECT COUNT(*) FROM lesson_notes WHERE MONTH(created_at) = MONTH(CURRENT_DATE()) AND YEAR(created_at) = YEAR(CURRENT_DATE())")->fetchColumn()
];

// AJAX istekleri için JSON yanıt
if (isset($_GET['edit']) || isset($_GET['view'])) {
    header('Content-Type: application/json');
    
    if (isset($_GET['edit'])) {
        $stmt = $pdo->prepare("SELECT * FROM lesson_notes WHERE id = ?");
        $stmt->execute([$_GET['edit']]);
        $note = $stmt->fetch();
        
        if ($note) {
            echo json_encode($note);
        } else {
            echo json_encode(['error' => 'Not bulunamadı']);
        }
    } elseif (isset($_GET['view'])) {
        $stmt = $pdo->prepare("SELECT * FROM lesson_notes WHERE id = ?");
        $stmt->execute([$_GET['view']]);
        $note = $stmt->fetch();
        
        if ($note) {
            $priority_text = ['low' => 'Düşük', 'medium' => 'Orta', 'high' => 'Yüksek'];
            $note['priority_text'] = $priority_text[$note['priority']];
            $note['created_at'] = formatDate($note['created_at']);
            echo json_encode($note);
        } else {
            echo json_encode(['error' => 'Not bulunamadı']);
        }
    }
    exit();
}

// Düzenleme için not getir
$edit_note = null;
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ders Notları - BAŞKENT 1910 GÜCÜ SK</title>
    <link rel="stylesheet" href="/admin/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <?php include '../includes/header.php'; ?>
    
    <div class="admin-container">
        <div class="page-header">
            <div class="page-title">
                <i class="fas fa-book"></i>
                <h2>Ders Notları</h2>
            </div>
            <div class="header-actions">
                <button class="btn btn-primary" onclick="openAddModal()">
                    <i class="fas fa-plus"></i> Yeni Not Ekle
                </button>
            </div>
        </div>

        <?php if (isset($success)): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <!-- İstatistikler -->
        <div class="lesson-notes-stats">
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-book"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['total']; ?></h3>
                        <p>Toplam Not</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-eye"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['published']; ?></h3>
                        <p>Yayınlanan</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-tags"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['categories']; ?></h3>
                        <p>Kategori</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-calendar"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo $stats['this_month']; ?></h3>
                        <p>Bu Ay</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filtreler -->
        <div class="filter-section">
            <form method="GET" class="filter-form">
                <div class="filter-group">
                    <input type="text" name="search" placeholder="Not ara..." value="<?php echo htmlspecialchars($search); ?>">
                </div>
                <div class="filter-group">
                    <select name="category">
                        <option value="">Tüm Kategoriler</option>
                        <?php foreach ($categories as $category): ?>
                            <option value="<?php echo htmlspecialchars($category); ?>" <?php echo $category_filter == $category ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($category); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="filter-group">
                    <select name="priority">
                        <option value="">Tüm Öncelikler</option>
                        <option value="low" <?php echo $priority_filter == 'low' ? 'selected' : ''; ?>>Düşük</option>
                        <option value="medium" <?php echo $priority_filter == 'medium' ? 'selected' : ''; ?>>Orta</option>
                        <option value="high" <?php echo $priority_filter == 'high' ? 'selected' : ''; ?>>Yüksek</option>
                    </select>
                </div>
                <button type="submit" class="btn btn-secondary">
                    <i class="fas fa-search"></i> Filtrele
                </button>
                <a href="lesson_notes.php" class="btn btn-outline">
                    <i class="fas fa-times"></i> Temizle
                </a>
            </form>
        </div>

        <!-- Ders Notları Listesi -->
        <div class="lesson-notes-list">
            <?php if (empty($lesson_notes)): ?>
                <div class="empty-state">
                    <i class="fas fa-book-open"></i>
                    <h3>Henüz ders notu yok</h3>
                    <p>İlk ders notunu eklemek için yukarıdaki butona tıklayın.</p>
                </div>
            <?php else: ?>
                <?php foreach ($lesson_notes as $note): ?>
                    <div class="lesson-note-card">
                        <div class="note-header">
                            <div class="note-title">
                                <h3><?php echo htmlspecialchars($note['title']); ?></h3>
                                <div class="note-meta">
                                    <span class="category"><?php echo htmlspecialchars($note['category']); ?></span>
                                    <span class="priority priority-<?php echo $note['priority']; ?>">
                                        <?php 
                                        $priority_text = ['low' => 'Düşük', 'medium' => 'Orta', 'high' => 'Yüksek'];
                                        echo $priority_text[$note['priority']];
                                        ?>
                                    </span>
                                    <?php if ($note['is_published']): ?>
                                        <span class="status-published">Yayında</span>
                                    <?php else: ?>
                                        <span class="status-draft">Taslak</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="note-actions">
                                <button class="btn btn-sm btn-primary" onclick="viewNote(<?php echo $note['id']; ?>)">
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button class="btn btn-sm btn-secondary" onclick="editNote(<?php echo $note['id']; ?>)">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-sm btn-danger" onclick="deleteNote(<?php echo $note['id']; ?>)">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </div>
                        <div class="note-content">
                            <?php if (!empty($note['image_path'])): ?>
                                <div class="note-image">
                                    <img src="<?php echo htmlspecialchars($note['image_path']); ?>" 
                                         alt="<?php echo htmlspecialchars($note['image_alt'] ?? $note['title']); ?>"
                                         onclick="viewImage('<?php echo htmlspecialchars($note['image_path']); ?>', '<?php echo htmlspecialchars($note['image_alt'] ?? $note['title']); ?>')">
                                </div>
                            <?php endif; ?>
                            <p><?php echo nl2br(htmlspecialchars(substr($note['content'], 0, 200))); ?><?php echo strlen($note['content']) > 200 ? '...' : ''; ?></p>
                        </div>
                        <div class="note-footer">
                            <div class="note-info">
                                <span><i class="fas fa-user"></i> <?php echo htmlspecialchars($note['author']); ?></span>
                                <span><i class="fas fa-calendar"></i> <?php echo date('d.m.Y H:i', strtotime($note['created_at'])); ?></span>
                            </div>
                            <?php if (!empty($note['tags'])): ?>
                                <div class="note-tags">
                                    <?php 
                                    $tags = explode(',', $note['tags']);
                                    foreach ($tags as $tag): 
                                        $tag = trim($tag);
                                        if (!empty($tag)):
                                    ?>
                                        <span class="tag"><?php echo htmlspecialchars($tag); ?></span>
                                    <?php 
                                        endif;
                                    endforeach; 
                                    ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

    <!-- Ekleme/Düzenleme Modal -->
    <div id="noteModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="modalTitle">
                    <i class="fas fa-plus"></i> Yeni Ders Notu
                </h3>
                <button class="modal-close" onclick="closeModal()">&times;</button>
            </div>
            <div class="modal-body">
                <form method="POST" id="noteForm" enctype="multipart/form-data">
                    <input type="hidden" name="action" id="formAction" value="add">
                    <input type="hidden" name="id" id="noteId">
                    
                    <div class="form-group">
                        <label for="title">Başlık *</label>
                        <input type="text" id="title" name="title" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="content">İçerik *</label>
                        <textarea id="content" name="content" rows="10" required></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label for="image">Resim</label>
                        <input type="file" id="image" name="image" accept="image/*" onchange="previewImage(this)">
                        <div id="imagePreview" class="image-preview" style="display: none;">
                            <img id="previewImg" src="" alt="Önizleme">
                            <button type="button" class="btn btn-sm btn-danger" onclick="removeImage()">Resmi Kaldır</button>
                        </div>
                        <small class="form-help">JPG, PNG, GIF, WebP formatları desteklenir. Maksimum 5MB.</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="image_alt">Resim Açıklaması</label>
                        <input type="text" id="image_alt" name="image_alt" placeholder="Resim için açıklama yazın">
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="category">Kategori</label>
                            <input type="text" id="category" name="category" value="Genel">
                        </div>
                        <div class="form-group">
                            <label for="author">Yazar</label>
                            <input type="text" id="author" name="author" value="Admin">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="priority">Öncelik</label>
                            <select id="priority" name="priority">
                                <option value="low">Düşük</option>
                                <option value="medium" selected>Orta</option>
                                <option value="high">Yüksek</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="tags">Etiketler (virgülle ayırın)</label>
                            <input type="text" id="tags" name="tags" placeholder="etiket1, etiket2, etiket3">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input type="checkbox" id="is_published" name="is_published" checked>
                            <span class="checkmark"></span>
                            Yayınla
                        </label>
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Kaydet
                        </button>
                        <button type="button" class="btn btn-secondary" onclick="closeModal()">
                            <i class="fas fa-times"></i> İptal
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Görüntüleme Modal -->
    <div id="viewModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="viewTitle">
                    <i class="fas fa-book"></i> Ders Notu
                </h3>
                <button class="modal-close" onclick="closeViewModal()">&times;</button>
            </div>
            <div class="modal-body">
                <div id="viewContent"></div>
            </div>
        </div>
    </div>

    <script>
        function openAddModal() {
            document.getElementById('modalTitle').innerHTML = '<i class="fas fa-plus"></i> Yeni Ders Notu';
            document.getElementById('formAction').value = 'add';
            document.getElementById('noteForm').reset();
            document.getElementById('noteId').value = '';
            document.getElementById('is_published').checked = true;
            document.getElementById('imagePreview').style.display = 'none';
            document.getElementById('noteModal').style.display = 'block';
        }

        function editNote(id) {
            // AJAX ile not bilgilerini getir
            fetch(`lesson_notes.php?edit=${id}`)
                .then(response => response.json())
                .then(data => {
                    document.getElementById('modalTitle').innerHTML = '<i class="fas fa-edit"></i> Ders Notunu Düzenle';
                    document.getElementById('formAction').value = 'edit';
                    document.getElementById('noteId').value = data.id;
                    document.getElementById('title').value = data.title;
                    document.getElementById('content').value = data.content;
                    document.getElementById('category').value = data.category;
                    document.getElementById('author').value = data.author;
                    document.getElementById('priority').value = data.priority;
                    document.getElementById('tags').value = data.tags;
                    document.getElementById('image_alt').value = data.image_alt || '';
                    document.getElementById('is_published').checked = data.is_published == 1;
                    
                    // Mevcut resmi göster
                    if (data.image_path) {
                        document.getElementById('previewImg').src = data.image_path;
                        document.getElementById('imagePreview').style.display = 'block';
                    } else {
                        document.getElementById('imagePreview').style.display = 'none';
                    }
                    
                    document.getElementById('noteModal').style.display = 'block';
                })
                .catch(error => {
                    alert('Not bilgileri alınamadı!');
                });
        }

        function viewNote(id) {
            // AJAX ile not içeriğini getir
            fetch(`lesson_notes.php?view=${id}`)
                .then(response => response.json())
                .then(data => {
                    document.getElementById('viewTitle').innerHTML = `<i class="fas fa-book"></i> ${data.title}`;
                    document.getElementById('viewContent').innerHTML = `
                        <div class="view-note">
                            <div class="view-meta">
                                <span class="category">${data.category}</span>
                                <span class="priority priority-${data.priority}">${data.priority_text}</span>
                                <span class="status-${data.is_published ? 'published' : 'draft'}">${data.is_published ? 'Yayında' : 'Taslak'}</span>
                            </div>
                            <div class="view-content">
                                ${data.content.replace(/\n/g, '<br>')}
                            </div>
                            <div class="view-footer">
                                <div class="view-info">
                                    <span><i class="fas fa-user"></i> ${data.author}</span>
                                    <span><i class="fas fa-calendar"></i> ${data.created_at}</span>
                                </div>
                                ${data.tags ? `<div class="view-tags">${data.tags.split(',').map(tag => `<span class="tag">${tag.trim()}</span>`).join('')}</div>` : ''}
                            </div>
                        </div>
                    `;
                    document.getElementById('viewModal').style.display = 'block';
                })
                .catch(error => {
                    alert('Not içeriği alınamadı!');
                });
        }

        function deleteNote(id) {
            if (confirm('Bu ders notunu silmek istediğinizden emin misiniz?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" value="${id}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        function closeModal() {
            document.getElementById('noteModal').style.display = 'none';
        }

        function closeViewModal() {
            document.getElementById('viewModal').style.display = 'none';
        }

        // Modal dışına tıklayınca kapat
        window.onclick = function(event) {
            const noteModal = document.getElementById('noteModal');
            const viewModal = document.getElementById('viewModal');
            if (event.target == noteModal) {
                noteModal.style.display = 'none';
            }
            if (event.target == viewModal) {
                viewModal.style.display = 'none';
            }
        }
    </script>
</body>
</html>
