<?php
// Attendance endpoints

// Authentication gerekli
$payload = authenticate();
$parent_id = $payload['parent_id'];

// RESTful yapı: action path'ten alınıyor (/api/v1/attendance/list)
// Eğer path'te action yoksa, GET parametresinden al (geriye dönük uyumluluk)
$action = $action ?? $_GET['action'] ?? '';

switch ($request_method) {
    case 'GET':
        // GET için varsayılan action list
        if (empty($action)) {
            $action = 'list';
        }
        
        switch ($action) {
            case 'list':
                // Parent'ın sporcularının yoklama bilgilerini listele
                $athlete_id = $_GET['athlete_id'] ?? '';
                $date_from = $_GET['date_from'] ?? '';
                $date_to = $_GET['date_to'] ?? '';
                $status = $_GET['status'] ?? '';
                
                $where_conditions = ["a.parent_id = ?"];
                $params = [$parent_id];
                
                if ($athlete_id) {
                    $where_conditions[] = "att.athlete_id = ?";
                    $params[] = $athlete_id;
                }
                
                if ($date_from) {
                    $where_conditions[] = "att.attendance_date >= ?";
                    $params[] = $date_from;
                }
                
                if ($date_to) {
                    $where_conditions[] = "att.attendance_date <= ?";
                    $params[] = $date_to;
                }
                
                if ($status) {
                    $where_conditions[] = "att.status = ?";
                    $params[] = $status;
                }
                
                $stmt = $pdo->prepare("
                    SELECT 
                        att.id,
                        att.attendance_date,
                        att.status,
                        att.notes,
                        att.created_at,
                        att.updated_at,
                        a.first_name,
                        a.last_name,
                        a.id as athlete_id
                    FROM attendance att
                    JOIN athletes a ON att.athlete_id = a.id
                    WHERE " . implode(' AND ', $where_conditions) . "
                    ORDER BY att.attendance_date DESC
                ");
                $stmt->execute($params);
                $attendance_records = $stmt->fetchAll();
                
                // Her yoklama için ek bilgiler
                foreach ($attendance_records as &$record) {
                    $record['athlete_name'] = $record['first_name'] . ' ' . $record['last_name'];
                    $record['formatted_date'] = date('d.m.Y', strtotime($record['attendance_date']));
                    $record['status_text'] = $record['status'] === 'present' ? 'Geldi' : 
                                           ($record['status'] === 'absent' ? 'Gelmedi' : 'Mazeretli');
                    $record['status_color'] = $record['status'] === 'present' ? 'success' : 
                                            ($record['status'] === 'absent' ? 'danger' : 'warning');
                }
                
                api_response($attendance_records, 200, 'Yoklama bilgileri başarıyla getirildi');
                break;
                
            case 'summary':
                // Yoklama özeti
                $athlete_id = $_GET['athlete_id'] ?? '';
                $month = $_GET['month'] ?? date('Y-m');
                $year = $_GET['year'] ?? date('Y');
                
                $where_conditions = ["a.parent_id = ?"];
                $params = [$parent_id];
                
                if ($athlete_id) {
                    $where_conditions[] = "att.athlete_id = ?";
                    $params[] = $athlete_id;
                }
                
                if ($month) {
                    $where_conditions[] = "DATE_FORMAT(att.attendance_date, '%Y-%m') = ?";
                    $params[] = $month;
                }
                
                if ($year) {
                    $where_conditions[] = "YEAR(att.attendance_date) = ?";
                    $params[] = $year;
                }
                
                // Genel yoklama istatistikleri
                $stmt = $pdo->prepare("
                    SELECT 
                        COUNT(*) as total_attendance,
                        SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_count,
                        SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_count,
                        SUM(CASE WHEN status = 'excused' THEN 1 ELSE 0 END) as excused_count
                    FROM attendance att
                    JOIN athletes a ON att.athlete_id = a.id
                    WHERE " . implode(' AND ', $where_conditions) . "
                ");
                $stmt->execute($params);
                $summary = $stmt->fetch();
                
                // Katılım oranı hesapla
                $attendance_rate = $summary['total_attendance'] > 0 ? 
                    round(($summary['present_count'] / $summary['total_attendance']) * 100, 1) : 0;
                
                // Günlük yoklama dağılımı
                $stmt = $pdo->prepare("
                    SELECT 
                        att.attendance_date,
                        COUNT(*) as total,
                        SUM(CASE WHEN att.status = 'present' THEN 1 ELSE 0 END) as present,
                        SUM(CASE WHEN att.status = 'absent' THEN 1 ELSE 0 END) as absent,
                        SUM(CASE WHEN att.status = 'excused' THEN 1 ELSE 0 END) as excused
                    FROM attendance att
                    JOIN athletes a ON att.athlete_id = a.id
                    WHERE " . implode(' AND ', $where_conditions) . "
                    GROUP BY att.attendance_date
                    ORDER BY att.attendance_date DESC
                    LIMIT 30
                ");
                $stmt->execute($params);
                $daily_data = $stmt->fetchAll();
                
                // Sporcu bazlı yoklama özeti
                $stmt = $pdo->prepare("
                    SELECT 
                        a.id,
                        a.first_name,
                        a.last_name,
                        COUNT(att.id) as total_attendance,
                        SUM(CASE WHEN att.status = 'present' THEN 1 ELSE 0 END) as present_count,
                        SUM(CASE WHEN att.status = 'absent' THEN 1 ELSE 0 END) as absent_count,
                        SUM(CASE WHEN att.status = 'excused' THEN 1 ELSE 0 END) as excused_count
                    FROM athletes a
                    LEFT JOIN attendance att ON a.id = att.athlete_id AND " . 
                    ($month ? "DATE_FORMAT(att.attendance_date, '%Y-%m') = ?" : "YEAR(att.attendance_date) = ?") . "
                    WHERE a.parent_id = ?
                    GROUP BY a.id, a.first_name, a.last_name
                    ORDER BY a.first_name, a.last_name
                ");
                $stmt->execute([$month ?: $year, $parent_id]);
                $athlete_summary = $stmt->fetchAll();
                
                // Her sporcu için katılım oranı hesapla
                foreach ($athlete_summary as &$athlete) {
                    $athlete['attendance_rate'] = $athlete['total_attendance'] > 0 ? 
                        round(($athlete['present_count'] / $athlete['total_attendance']) * 100, 1) : 0;
                    $athlete['athlete_name'] = $athlete['first_name'] . ' ' . $athlete['last_name'];
                }
                
                $result = [
                    'summary' => [
                        'total_attendance' => (int)$summary['total_attendance'],
                        'present_count' => (int)$summary['present_count'],
                        'absent_count' => (int)$summary['absent_count'],
                        'excused_count' => (int)$summary['excused_count'],
                        'attendance_rate' => $attendance_rate
                    ],
                    'daily_data' => $daily_data,
                    'athlete_summary' => $athlete_summary
                ];
                
                api_response($result, 200, 'Yoklama özeti başarıyla getirildi');
                break;
                
            case 'calendar':
                // Takvim görünümü
                $month = $_GET['month'] ?? date('Y-m');
                $athlete_id = $_GET['athlete_id'] ?? '';
                
                $where_conditions = ["a.parent_id = ?"];
                $params = [$parent_id];
                
                if ($athlete_id) {
                    $where_conditions[] = "att.athlete_id = ?";
                    $params[] = $athlete_id;
                }
                
                if ($month) {
                    $where_conditions[] = "DATE_FORMAT(att.attendance_date, '%Y-%m') = ?";
                    $params[] = $month;
                }
                
                $stmt = $pdo->prepare("
                    SELECT 
                        att.attendance_date,
                        att.status,
                        att.notes,
                        a.first_name,
                        a.last_name,
                        a.id as athlete_id
                    FROM attendance att
                    JOIN athletes a ON att.athlete_id = a.id
                    WHERE " . implode(' AND ', $where_conditions) . "
                    ORDER BY att.attendance_date, a.first_name, a.last_name
                ");
                $stmt->execute($params);
                $attendance_records = $stmt->fetchAll();
                
                // Günlere göre grupla
                $calendar = [];
                foreach ($attendance_records as $record) {
                    $date = $record['attendance_date'];
                    if (!isset($calendar[$date])) {
                        $calendar[$date] = [
                            'date' => $date,
                            'formatted_date' => date('d.m.Y', strtotime($date)),
                            'day_name' => date('l', strtotime($date)),
                            'records' => []
                        ];
                    }
                    
                    $record['athlete_name'] = $record['first_name'] . ' ' . $record['last_name'];
                    $record['status_text'] = $record['status'] === 'present' ? 'Geldi' : 
                                           ($record['status'] === 'absent' ? 'Gelmedi' : 'Mazeretli');
                    $record['status_color'] = $record['status'] === 'present' ? 'success' : 
                                            ($record['status'] === 'absent' ? 'danger' : 'warning');
                    
                    $calendar[$date]['records'][] = $record;
                }
                
                api_response(array_values($calendar), 200, 'Takvim verileri başarıyla getirildi');
                break;
                
            case 'detail':
                // Belirli bir yoklama detayı
                $attendance_id = $_GET['id'] ?? '';
                
                if (!$attendance_id) {
                    api_error('Yoklama ID gerekli', 400);
                }
                
                $stmt = $pdo->prepare("
                    SELECT 
                        att.*,
                        a.first_name,
                        a.last_name,
                        a.id as athlete_id
                    FROM attendance att
                    JOIN athletes a ON att.athlete_id = a.id
                    WHERE att.id = ? AND a.parent_id = ?
                ");
                $stmt->execute([$attendance_id, $parent_id]);
                $attendance = $stmt->fetch();
                
                if (!$attendance) {
                    api_error('Yoklama kaydı bulunamadı', 404);
                }
                
                $attendance['athlete_name'] = $attendance['first_name'] . ' ' . $attendance['last_name'];
                $attendance['formatted_date'] = date('d.m.Y', strtotime($attendance['attendance_date']));
                $attendance['status_text'] = $attendance['status'] === 'present' ? 'Geldi' : 
                                           ($attendance['status'] === 'absent' ? 'Gelmedi' : 'Mazeretli');
                $attendance['status_color'] = $attendance['status'] === 'present' ? 'success' : 
                                            ($attendance['status'] === 'absent' ? 'danger' : 'warning');
                
                api_response($attendance, 200, 'Yoklama detayları başarıyla getirildi');
                break;
                
            default:
                api_error('Geçersiz işlem', 400);
        }
        break;
        
    default:
        api_error('Desteklenmeyen HTTP metodu', 405);
}
?>
