<?php
// Profile endpoints

// Authentication gerekli
$payload = authenticate();
$parent_id = $payload['parent_id'];
$user_id = $payload['user_id'];

// RESTful yapı: action path'ten alınıyor (/api/v1/profile/info)
// Eğer path'te action yoksa, GET parametresinden al (geriye dönük uyumluluk)
$action = $action ?? $_GET['action'] ?? '';

switch ($request_method) {
    case 'GET':
        // GET için varsayılan action info
        if (empty($action)) {
            $action = 'info';
        }
        
        switch ($action) {
            case 'info':
                // Parent profil bilgileri
                $stmt = $pdo->prepare("
                    SELECT 
                        p.*,
                        u.username,
                        u.email,
                        u.full_name,
                        u.created_at as user_created_at,
                        u.last_login
                    FROM parents p
                    JOIN users u ON p.user_id = u.id
                    WHERE p.id = ?
                ");
                $stmt->execute([$parent_id]);
                $parent = $stmt->fetch();
                
                if (!$parent) {
                    api_error('Profil bulunamadı', 404);
                }
                
                // Sporcu sayısı
                // NOT: athletes.parent_id hem users.id hem de parents.id olabilir
                $user_id = $payload['user_id'] ?? null;
                $stmt = $pdo->prepare("SELECT COUNT(*) as athlete_count FROM athletes WHERE parent_id = ? OR (parent_id = ? AND ? IS NOT NULL)");
                $stmt->execute([$parent_id, $user_id, $user_id]);
                $athlete_count = $stmt->fetch()['athlete_count'];
                
                // Son aktivite (trainings tablosu yoksa sadece payments ve attendance)
                $recent_activities = [];
                
                    // Payments
                    try {
                        $stmt = $pdo->prepare("
                            SELECT 
                                'payment' as type,
                                p.amount,
                                p.payment_date as activity_date,
                                p.status
                            FROM payments p
                            JOIN athletes a ON p.athlete_id = a.id
                            WHERE a.parent_id = ? OR (a.parent_id = ? AND ? IS NOT NULL)
                            ORDER BY p.payment_date DESC
                            LIMIT 5
                        ");
                        $stmt->execute([$parent_id, $user_id, $user_id]);
                    $payments = $stmt->fetchAll();
                    $recent_activities = array_merge($recent_activities, $payments);
                } catch (PDOException $e) {
                    // Payments tablosu yoksa atla
                }
                
                // Trainings (tablo varsa)
                try {
                    $test_stmt = $pdo->query("SHOW TABLES LIKE 'trainings'");
                    if ($test_stmt->rowCount() > 0) {
                            $stmt = $pdo->prepare("
                                SELECT 
                                    'training' as type,
                                    NULL as amount,
                                    t.training_date as activity_date,
                                    t.status
                                FROM trainings t
                                JOIN athletes a ON t.athlete_id = a.id
                                WHERE a.parent_id = ? OR (a.parent_id = ? AND ? IS NOT NULL)
                                ORDER BY t.training_date DESC
                                LIMIT 5
                            ");
                            $stmt->execute([$parent_id, $user_id, $user_id]);
                        $trainings = $stmt->fetchAll();
                        $recent_activities = array_merge($recent_activities, $trainings);
                    }
                } catch (PDOException $e) {
                    // Trainings tablosu yoksa atla
                }
                
                // Attendance
                try {
                        $stmt = $pdo->prepare("
                            SELECT 
                                'attendance' as type,
                                NULL as amount,
                                att.training_date as activity_date,
                                att.status
                            FROM attendance att
                            JOIN athletes a ON att.athlete_id = a.id
                            WHERE a.parent_id = ? OR (a.parent_id = ? AND ? IS NOT NULL)
                            ORDER BY att.training_date DESC
                            LIMIT 5
                        ");
                        $stmt->execute([$parent_id, $user_id, $user_id]);
                    $attendance = $stmt->fetchAll();
                    $recent_activities = array_merge($recent_activities, $attendance);
                } catch (PDOException $e) {
                    // Attendance tablosu yoksa atla
                }
                
                // Tarihe göre sırala ve ilk 10'unu al
                usort($recent_activities, function($a, $b) {
                    return strtotime($b['activity_date']) - strtotime($a['activity_date']);
                });
                $recent_activities = array_slice($recent_activities, 0, 10);
                
                // Her aktivite için formatla
                foreach ($recent_activities as &$activity) {
                    $activity['formatted_date'] = date('d.m.Y', strtotime($activity['activity_date']));
                    $activity['type_text'] = $activity['type'] === 'payment' ? 'Ödeme' : 
                                           ($activity['type'] === 'training' ? 'Antrenman' : 'Yoklama');
                    $activity['status_text'] = $activity['status'] === 'paid' ? 'Ödendi' : 
                                             ($activity['status'] === 'pending' ? 'Bekliyor' : 
                                             ($activity['status'] === 'overdue' ? 'Gecikmiş' : 
                                             ($activity['status'] === 'present' ? 'Geldi' : 
                                             ($activity['status'] === 'absent' ? 'Gelmedi' : 
                                             ($activity['status'] === 'excused' ? 'Mazeretli' : 
                                             ($activity['status'] === 'scheduled' ? 'Planlandı' : 
                                             ($activity['status'] === 'completed' ? 'Tamamlandı' : 'İptal Edildi')))))));
                }
                
                $profile = [
                    'id' => $parent['id'],
                    'username' => $parent['username'],
                    'email' => $parent['email'],
                    'full_name' => $parent['full_name'],
                    'phone' => $parent['phone'],
                    'address' => $parent['address'],
                    'created_at' => $parent['user_created_at'],
                    'last_login' => $parent['last_login'],
                    'athlete_count' => (int)$athlete_count,
                    'recent_activities' => $recent_activities
                ];
                
                api_response($profile, 200, 'Profil bilgileri başarıyla getirildi');
                break;
                
            case 'stats':
                // Parent istatistikleri
                $stmt = $pdo->prepare("
                    SELECT 
                        COUNT(DISTINCT a.id) as total_athletes,
                        COUNT(DISTINCT p.id) as total_payments,
                        COUNT(DISTINCT t.id) as total_trainings,
                        COUNT(DISTINCT att.id) as total_attendance
                    FROM athletes a
                    LEFT JOIN payments p ON a.id = p.athlete_id
                    LEFT JOIN trainings t ON a.id = t.athlete_id
                    LEFT JOIN attendance att ON a.id = att.athlete_id
                    WHERE a.parent_id = ? OR (a.parent_id = ? AND ? IS NOT NULL)
                ");
                $stmt->execute([$parent_id, $user_id, $user_id]);
                $stats = $stmt->fetch();
                
                // Ödeme istatistikleri
                $stmt = $pdo->prepare("
                    SELECT 
                        SUM(p.amount) as total_paid,
                        COUNT(CASE WHEN p.status = 'paid' THEN 1 END) as paid_count,
                        COUNT(CASE WHEN p.status = 'pending' THEN 1 END) as pending_count,
                        COUNT(CASE WHEN p.status = 'overdue' THEN 1 END) as overdue_count
                    FROM payments p
                    JOIN athletes a ON p.athlete_id = a.id
                    WHERE a.parent_id = ? OR (a.parent_id = ? AND ? IS NOT NULL)
                ");
                $stmt->execute([$parent_id, $user_id, $user_id]);
                $payment_stats = $stmt->fetch();
                
                // Yoklama istatistikleri
                $stmt = $pdo->prepare("
                    SELECT 
                        COUNT(*) as total_attendance,
                        SUM(CASE WHEN att.status = 'present' THEN 1 ELSE 0 END) as present_count,
                        SUM(CASE WHEN att.status = 'absent' THEN 1 ELSE 0 END) as absent_count,
                        SUM(CASE WHEN att.status = 'excused' THEN 1 ELSE 0 END) as excused_count
                    FROM attendance att
                    JOIN athletes a ON att.athlete_id = a.id
                    WHERE a.parent_id = ? OR (a.parent_id = ? AND ? IS NOT NULL)
                ");
                $stmt->execute([$parent_id, $user_id, $user_id]);
                $attendance_stats = $stmt->fetch();
                
                // Katılım oranı hesapla
                $attendance_rate = $attendance_stats['total_attendance'] > 0 ? 
                    round(($attendance_stats['present_count'] / $attendance_stats['total_attendance']) * 100, 1) : 0;
                
                $result = [
                    'athletes' => [
                        'total' => (int)$stats['total_athletes']
                    ],
                    'payments' => [
                        'total' => (int)$stats['total_payments'],
                        'total_amount' => (float)$payment_stats['total_paid'],
                        'paid_count' => (int)$payment_stats['paid_count'],
                        'pending_count' => (int)$payment_stats['pending_count'],
                        'overdue_count' => (int)$payment_stats['overdue_count']
                    ],
                    'trainings' => [
                        'total' => (int)$stats['total_trainings']
                    ],
                    'attendance' => [
                        'total' => (int)$attendance_stats['total_attendance'],
                        'present_count' => (int)$attendance_stats['present_count'],
                        'absent_count' => (int)$attendance_stats['absent_count'],
                        'excused_count' => (int)$attendance_stats['excused_count'],
                        'attendance_rate' => $attendance_rate
                    ]
                ];
                
                api_response($result, 200, 'Profil istatistikleri başarıyla getirildi');
                break;
                
            default:
                api_error('Geçersiz işlem', 400);
        }
        break;
        
    case 'PUT':
        // PUT için varsayılan action update
        if (empty($action)) {
            $action = 'update';
        }
        
        switch ($action) {
            case 'update':
                // Profil güncelleme
                $input = json_decode(file_get_contents('php://input'), true);
                
                if (!$input) {
                    api_error('Geçersiz veri', 400);
                }
                
                // Güncellenebilir alanlar
                $allowed_fields = ['phone', 'address'];
                $update_fields = [];
                $update_values = [];
                
                foreach ($allowed_fields as $field) {
                    if (isset($input[$field])) {
                        $update_fields[] = "$field = ?";
                        $update_values[] = $input[$field];
                    }
                }
                
                if (empty($update_fields)) {
                    api_error('Güncellenecek alan bulunamadı', 400);
                }
                
                $update_values[] = $parent_id;
                
                $stmt = $pdo->prepare("
                    UPDATE parents 
                    SET " . implode(', ', $update_fields) . ", updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute($update_values);
                
                api_response(null, 200, 'Profil başarıyla güncellendi');
                break;
                
            case 'change_password':
                // Şifre değiştirme
                $input = json_decode(file_get_contents('php://input'), true);
                
                if (!$input || !isset($input['current_password']) || !isset($input['new_password'])) {
                    api_error('Mevcut şifre ve yeni şifre gerekli', 400);
                }
                
                $current_password = $input['current_password'];
                $new_password = $input['new_password'];
                
                // Mevcut şifreyi kontrol et
                $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $user = $stmt->fetch();
                
                if (!$user || !password_verify($current_password, $user['password'])) {
                    api_error('Mevcut şifre yanlış', 400);
                }
                
                // Yeni şifreyi hashle ve güncelle
                $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("
                    UPDATE users 
                    SET password = ?, updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$hashed_password, $user_id]);
                
                // Şifre geçmişine ekle
                $stmt = $pdo->prepare("
                    INSERT INTO password_history (user_id, password_hash, changed_at) 
                    VALUES (?, ?, NOW())
                ");
                $stmt->execute([$user_id, $hashed_password]);
                
                api_response(null, 200, 'Şifre başarıyla değiştirildi');
                break;
                
            default:
                api_error('Geçersiz işlem', 400);
        }
        break;
        
    default:
        api_error('Desteklenmeyen HTTP metodu', 405);
}
?>
