<?php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// OPTIONS isteği için CORS preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Hata raporlama - JSON API için hataları JSON formatında döndür
error_reporting(E_ALL);
ini_set('display_errors', 0); // HTML hataları kapat
ini_set('log_errors', 1); // Log'a yaz

// Hata yakalama fonksiyonu
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    // JSON header'ı zaten gönderilmiş olabilir, bu yüzden kontrol et
    if (!headers_sent()) {
        header('Content-Type: application/json; charset=utf-8');
    }
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Sunucu hatası: ' . $errstr,
        'timestamp' => date('Y-m-d H:i:s'),
        'data' => null
    ], JSON_UNESCAPED_UNICODE);
    exit();
});

// Fatal error yakalama
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        if (!headers_sent()) {
            header('Content-Type: application/json; charset=utf-8');
        }
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Sunucu hatası: ' . $error['message'],
            'timestamp' => date('Y-m-d H:i:s'),
            'data' => null
        ], JSON_UNESCAPED_UNICODE);
        exit();
    }
});

// Veritabanı bağlantısı
try {
    require_once __DIR__ . '/../config/database.php';
    
    // Veritabanı bağlantısı kontrolü
    if (!isset($pdo) || !$pdo) {
        throw new Exception('Veritabanı bağlantısı kurulamadı');
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Veritabanı bağlantı hatası: ' . $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s'),
        'data' => null
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

// JWT sınıfı
class JWT {
    private static $secret_key = 'baskent1910guksk_secret_key_2024';
    
    public static function encode($payload) {
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
        $payload = json_encode($payload);
        
        $base64_header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64_payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
        
        $signature = hash_hmac('sha256', $base64_header . "." . $base64_payload, self::$secret_key, true);
        $base64_signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        return $base64_header . "." . $base64_payload . "." . $base64_signature;
    }
    
    public static function decode($jwt) {
        $token_parts = explode('.', $jwt);
        if (count($token_parts) !== 3) {
            return false;
        }
        
        $header = base64_decode(str_replace(['-', '_'], ['+', '/'], $token_parts[0]));
        $payload = base64_decode(str_replace(['-', '_'], ['+', '/'], $token_parts[1]));
        $signature = base64_decode(str_replace(['-', '_'], ['+', '/'], $token_parts[2]));
        
        $expected_signature = hash_hmac('sha256', $token_parts[0] . "." . $token_parts[1], self::$secret_key, true);
        
        if (!hash_equals($signature, $expected_signature)) {
            return false;
        }
        
        return json_decode($payload, true);
    }
}

// Rate limiting
class RateLimiter {
    private static $requests = [];
    
    public static function check($ip, $limit = 100, $window = 3600) {
        $now = time();
        $key = $ip . '_' . floor($now / $window);
        
        if (!isset(self::$requests[$key])) {
            self::$requests[$key] = 0;
        }
        
        if (self::$requests[$key] >= $limit) {
            return false;
        }
        
        self::$requests[$key]++;
        return true;
    }
}

// API yanıt fonksiyonu
function api_response($data = null, $status = 200, $message = '') {
    http_response_code($status);
    
    $response = [
        'success' => $status >= 200 && $status < 300,
        'message' => $message,
        'timestamp' => date('Y-m-d H:i:s'),
        'data' => $data
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit();
}

// Hata yanıtı
function api_error($message, $status = 400) {
    api_response(null, $status, $message);
}

// Rate limiting kontrolü
$client_ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
if (!RateLimiter::check($client_ip)) {
    api_error('Çok fazla istek gönderildi. Lütfen daha sonra tekrar deneyin.', 429);
}

// Route yönetimi
$request_method = $_SERVER['REQUEST_METHOD'];
$request_uri = $_SERVER['REQUEST_URI'];
$path = parse_url($request_uri, PHP_URL_PATH);

// Path'i temizle ve parçalara ayır
$path = trim($path, '/');
$path_parts = array_filter(explode('/', $path)); // Boş elemanları temizle
$path_parts = array_values($path_parts); // İndeksleri yeniden düzenle

// Debug için (production'da kaldırılabilir)
// error_log("Request URI: " . $request_uri);
// error_log("Path: " . $path);
// error_log("Path parts: " . json_encode($path_parts));

// Minimum kontrol: en az 'api' ve 'v1' olmalı
if (count($path_parts) < 2) {
    api_error('Geçersiz API endpoint. URL formatı: /api/v1/endpoint/action', 404);
}

// İlk eleman 'api' olmalı
if ($path_parts[0] !== 'api') {
    api_error('Geçersiz API endpoint. URL formatı: /api/v1/endpoint/action', 404);
}

$version = $path_parts[1] ?? 'v1';
$endpoint = $path_parts[2] ?? '';
$action = $path_parts[3] ?? ''; // RESTful yapı için action path'ten alınıyor

// API versiyonu kontrolü
if ($version !== 'v1') {
    api_error('Desteklenmeyen API versiyonu. Sadece v1 destekleniyor.', 400);
}

// Endpoint boşsa hata ver
if (empty($endpoint)) {
    api_error('Endpoint belirtilmedi. Örnek: /api/v1/profile/info', 404);
}

// Authentication middleware
function authenticate() {
    $headers = getallheaders();
    $auth_header = $headers['Authorization'] ?? $headers['authorization'] ?? '';
    
    if (!$auth_header || !preg_match('/Bearer\s+(.*)$/i', $auth_header, $matches)) {
        api_error('Authorization token gerekli', 401);
    }
    
    $token = $matches[1];
    $payload = JWT::decode($token);
    
    if (!$payload) {
        api_error('Geçersiz token', 401);
    }
    
    // Token süresi kontrolü
    if (isset($payload['exp']) && $payload['exp'] < time()) {
        api_error('Token süresi dolmuş', 401);
    }
    
    return $payload;
}

// Test endpoint (geliştirme için)
if ($endpoint === 'test') {
    api_response([
        'message' => 'API çalışıyor!',
        'request_uri' => $request_uri,
        'path' => $path,
        'path_parts' => $path_parts,
        'version' => $version,
        'endpoint' => $endpoint,
        'action' => $action,
        'method' => $request_method
    ], 200, 'Test başarılı');
}

// Endpoint routing
switch ($endpoint) {
    case 'auth':
        include __DIR__ . '/endpoints/auth.php';
        break;
        
    case 'athletes':
        include __DIR__ . '/endpoints/athletes.php';
        break;
        
    case 'payments':
        include __DIR__ . '/endpoints/payments.php';
        break;
        
    case 'trainings':
        include __DIR__ . '/endpoints/trainings.php';
        break;
        
    case 'attendance':
        include __DIR__ . '/endpoints/attendance.php';
        break;
        
    case 'notifications':
        include __DIR__ . '/endpoints/notifications.php';
        break;
        
    case 'profile':
        include __DIR__ . '/endpoints/profile.php';
        break;
        
    case 'lesson-notes':
        include __DIR__ . '/endpoints/lesson_notes.php';
        break;
        
    default:
        api_error('Endpoint bulunamadı', 404);
}

// Varsayılan 404
api_error('API endpoint bulunamadı', 404);
?>
